﻿using AssetWorx.Entities;
using AssetWorx.Plugin;
using Newtonsoft.Json;
using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Text;

namespace SignaWorksAlarmPlugin
{
    public class SignaWorksCustomAction : ICustomAction
    {
        public string Name => "SignaWorks Alarm Plugin Custom Action";
        private ReaderAlarm[] readerAlarms;
        private Dictionary<SignaWorksClient, ReaderAlarm> clientToReaderAlarm = new Dictionary<SignaWorksClient, ReaderAlarm>();
        private List<SignaWorksClient> signaWorksClients = new List<SignaWorksClient>();
        private Dictionary<string, LinkedList<DateTime>> tagIDCounts = new Dictionary<string, LinkedList<DateTime>>();
        bool debugMode;
        Logger<SignaWorksCustomAction> logger;
        public SignaWorksCustomAction()
        {            
        }
        
        public void Dispose()
        {
            foreach (var client in signaWorksClients)
            {
                client.Dispose();
            }
        }

        public void DoAction(Event ev, Asset asset)
        {
            try
            {
                string tagId = string.IsNullOrWhiteSpace(asset.VTagID) ? asset.RfidTag : asset.VTagID;
                if (string.IsNullOrWhiteSpace(tagId))
                    return;
                foreach (var client in signaWorksClients)
                {
                    if (client.Name == ev.ReaderName)
                    {
                        if (client.IsConnected)
                        {
                            bool exceeded = ThresholdExceeded(ev.ReaderName, tagId);
                            if (exceeded)
                                client.StartAlarm();
                        }
                        else
                        {
                            logger.Write(string.Format("Alarm {0} with Address {1} is not connected.", client.Name, client.IpAddress));
                        }
                    }
                }
            }
            catch(Exception exc)
            {
                logger.WriteException("SignaWorksCustomAction.DoAction exception: "+exc.Message);
            }
        }

        public void Initialize()
        {
            string documentsPath = Environment.GetFolderPath(Environment.SpecialFolder.CommonDocuments);
            string configPath = Path.Combine(documentsPath, "SignaWorksAlarmPlugin.json");
            //go through a configuration file and read the number of 
            if (!File.Exists(configPath))
                readerAlarms = new ReaderAlarm[0];
            else
            {
                string fileContent = File.ReadAllText(configPath);
                try
                {
                    Configuration configuration = JsonConvert.DeserializeObject<Configuration>(fileContent);
                    debugMode = configuration.DebugMode;
                    logger = new Logger<SignaWorksCustomAction>(debugMode);
                    readerAlarms = configuration.ReaderAlarms;
                    foreach (var readerAlarm in readerAlarms)
                    {
                        SignaWorksClient tempClient = new SignaWorksClient(readerAlarm.ReaderName, readerAlarm.IpAddress, readerAlarm.Port, readerAlarm.AlarmDuration, debugMode);
                        signaWorksClients.Add(tempClient);
                        clientToReaderAlarm.Add(tempClient, readerAlarm);
                    }
                }
                catch (Exception exc)
                {
                    if (logger != null)
                        logger.WriteException("SignaWorksCustomAction exception: " + exc.Message);
                }
            }
        }

        private bool ThresholdExceeded(string readerName, string tagId)
        {
            ReaderAlarm readerAlarm = readerAlarms.FirstOrDefault(p => p.ReaderName == readerName);
            if (readerAlarm == null)
                return false;
            if (!tagIDCounts.ContainsKey(tagId))
                tagIDCounts.Add(tagId, new LinkedList<DateTime>());
            LinkedList<DateTime> dates = tagIDCounts[tagId];
            int recentCount = 1;
            var current = dates.Last;
            while (current != null)
            {
                DateTime currentDate = current.Value;
                double minutes = (DateTime.Now - currentDate).TotalMinutes;
                System.Diagnostics.Debug.WriteLine("tag id=" + tagId + "DateTime=" + currentDate.ToString() + "minutes=" + minutes);
                var temp = current;
                current = current.Previous;
                if (minutes > readerAlarm.RecentTimeoutMinutes)
                    dates.Remove(temp);
                else
                    recentCount++;
            }
            dates.AddFirst(DateTime.Now);
            System.Diagnostics.Debug.WriteLine("recentCount=" + recentCount + " and threshold=" + readerAlarm.RecentThreshold);
            if (recentCount >= readerAlarm.RecentThreshold)
                return true;
            else
                return false;
        }
        
    }
}
