﻿using System;
using System.Collections.Generic;
using System.Text;
using System.Timers;

namespace SignaWorksAlarmPlugin
{
    public class SignaWorksClient : IDisposable
    {
        MySocketClient socketClient;
        Timer alarmTimeoutTimer;
        Timer checkConnectionTimer;
        Timer reconnectTimer;

        public string Name { get; }
        public string IpAddress { get; }
        public int Port { get; }
        public int AlarmDuration { get; }
        public bool IsConnected { get; set; } = false;
        public Logger<SignaWorksClient> Logger { get; }
        public bool DebugMode { get; set; } = false;
        public SignaWorksClient(string name, string ipAddress, int port, int alarmDuration, bool debugMode)
        {
            Name = name;
            IpAddress = ipAddress;
            Port = port;
            AlarmDuration = alarmDuration;
            DebugMode = debugMode;
            Logger = new Logger<SignaWorksClient>(debugMode);
            reconnectTimer = new Timer(15000);
            socketClient = new MySocketClient(debugMode);
            reconnectTimer.Elapsed += ReconnectTimer_Elapsed;
            reconnectTimer.Start();
            checkConnectionTimer = new Timer(15000);
            checkConnectionTimer.Elapsed += CheckConnectionTimer_Elapsed;
            checkConnectionTimer.Start();
            alarmTimeoutTimer = new Timer(AlarmDuration * 1000);
            alarmTimeoutTimer.Elapsed += TimeoutTimer_Elapsed;
            socketClient.Disconnected += SocketClient_Disconnected;
            socketClient.DataRead += SocketClient_DataRead;
            socketClient.InfoMessage += SocketClient_InfoMessage;            
        }

        private void ReconnectTimer_Elapsed(object sender, ElapsedEventArgs e)
        {
            if (!IsConnected)
            {
                Logger.Write("ReconnectTimer TryReconnect. IsConnected=" + IsConnected);
                TryConnect();
            }
        }

        private void CheckConnectionTimer_Elapsed(object sender, ElapsedEventArgs e)
        {
            if (IsConnected)
                CheckAlarm();
        }

        private void TimeoutTimer_Elapsed(object sender, ElapsedEventArgs e)
        {
            alarmTimeoutTimer.Stop();
            StopAlarm();
        }

        private void TryConnect()
        {
            IsConnected = socketClient.Connect(IpAddress, Port);
            if (IsConnected)
                StopAlarm();
        }

        private void SocketClient_InfoMessage(object sender, string e)
        {
            Logger.Write("InfoMessage:" + e);
        }

        private void SocketClient_DataRead(object sender, byte[] data)
        {
            //Logger.Write("LineRead:" + ByteArrayToString(data));
            if (data.Length == 10)
            {
                byte soundGroup = data[0];
                byte rLamp = data[1];
                byte soundChannel = data[7];
            }
        }

        private void SocketClient_Disconnected(object sender, EventArgs e)
        {
            IsConnected = false;
            Logger.Write("Disconnected:");
        }

        public void StartAlarm()
        {
            try
            {
                if (IsConnected)
                {
                    byte[] bytes = new byte[8];
                    bytes[0] = 0x57;
                    bytes[1] = 0x05;
                    bytes[2] = 0x02;
                    bytes[3] = 0x00;
                    bytes[4] = 0x00;
                    bytes[5] = 0x00;
                    bytes[6] = 0x00;
                    bytes[7] = 0x01;
                    socketClient.Write(bytes);
                }
                alarmTimeoutTimer.Start();
            }
            catch (Exception exc)
            {
                Logger.WriteException("Start Alarm Exception:" + exc.Message);
            }
            /*
            Socket soc = new Socket(AddressFamily.InterNetwork, SocketType.Stream, ProtocolType.Tcp);
            System.Net.IPAddress ipAdd = System.Net.IPAddress.Parse(ipAddress);
            System.Net.IPEndPoint remoteEP = new IPEndPoint(ipAdd, 3456);
            */

        }

        public void StopAlarm()
        {
            try
            {
                if (IsConnected)
                {
                    byte[] bytes = new byte[8];
                    bytes[0] = 0x57;
                    bytes[1] = 0x00;
                    bytes[2] = 0x00;
                    bytes[3] = 0x00;
                    bytes[4] = 0x00;
                    bytes[5] = 0x00;
                    bytes[6] = 0x00;
                    bytes[7] = 0x00;
                    socketClient.Write(bytes);
                }
            }
            catch (Exception exc)
            {
                Logger.WriteException("Stop Alarm Exception:" + exc.Message);
            }
        }

        public void CheckAlarm()
        {
            try
            {
                if (IsConnected)
                {
                    byte[] bytes = new byte[8];
                    bytes[0] = 0x52;
                    bytes[1] = 0x00;
                    bytes[2] = 0x00;
                    bytes[3] = 0x00;
                    bytes[4] = 0x00;
                    bytes[5] = 0x00;
                    bytes[6] = 0x00;
                    bytes[7] = 0x00;
                    socketClient.Write(bytes);
                }
            }
            catch (Exception exc)
            {
                Logger.WriteException("Got CheckAlarm Exception: " + exc.Message);
            }
        }

        public void Dispose()
        {
            alarmTimeoutTimer.Elapsed -= TimeoutTimer_Elapsed;
            socketClient.Disconnected -= SocketClient_Disconnected;
            socketClient.DataRead -= SocketClient_DataRead;
            socketClient.InfoMessage -= SocketClient_InfoMessage;
            socketClient.Disconnect();
        }

        public static string ByteArrayToString(byte[] ba)
        {
            StringBuilder hex = new StringBuilder(ba.Length * 2);
            foreach (byte b in ba)
                hex.AppendFormat("{0:x2}", b);
            return hex.ToString();
        }
    }
}
