﻿CREATE TABLE public.alarm
(
  id serial NOT NULL,
  alarmtime timestamp with time zone NOT NULL DEFAULT now(),
  rfidtag text,
  readername text,
  readerlocation text,
  assetname text,
  assetdescription text,
  alertingactionname text,
  eventtype text,
  thresholdtype text,
  eventdetails text,
  antennanumber text,
  rssi text,
  popuprequested boolean,
  vtagid text,
  lastmodifiedby text,
  CONSTRAINT pk_alarm_id PRIMARY KEY (id)
)
WITH (
  OIDS=FALSE
);
ALTER TABLE public.alarm
  OWNER TO postgres;

CREATE TABLE public.alarmack
(
  id serial NOT NULL,
  name text,
  tag text,
  eventtype text,
  thresholdtype text,
  assetname text,
  assetdescription text,
  acktime timestamp with time zone NOT NULL DEFAULT now(),
  lastmodifiedby text,
  CONSTRAINT pk_alarmack_id PRIMARY KEY (id)
)
WITH (
  OIDS=FALSE
);
ALTER TABLE public.alarmack
  OWNER TO postgres;

CREATE TABLE public.alertingaction
(
  id serial NOT NULL,
  name text,
  adminpopupalert boolean,
  checkin boolean,
  checkout boolean,
  logalarm boolean,
  onlyifcheckedin boolean,
  autoack boolean,
  pluginactions text,
  changelocation boolean,
  smtpemail boolean,
  gpo1 boolean,
  gpo1state text,
  gpo1seconds integer,
  gpo2 boolean,
  gpo2state text,
  gpo2seconds integer,
  lastmodifiedby text,
  CONSTRAINT pk_alertingaction_id PRIMARY KEY (id)
)
WITH (
  OIDS=FALSE
);
ALTER TABLE public.alertingaction
  OWNER TO postgres;

CREATE TABLE public.applicationsetting
(
  id serial NOT NULL,
  version text,
  type text,
  licensekey text,
  company text DEFAULT 'ACMEE'::text,
  lastrfidserialnumber integer NOT NULL DEFAULT 1,
  temperatureformat text DEFAULT 'Fahrenheit'::text,
  lastmodifiedby text,
  usemeters boolean DEFAULT false,
  assetdisplaycolumns text DEFAULT 'name,vTagID,rfidTag,locationName,description,created'::text,
  vtagpopupdisplayrows text DEFAULT 'name,vTagID,vTagLastSeen,batteryLevel,description'::text,
  cmdmaprefresh integer DEFAULT 15,
  cmdactivetimeout integer DEFAULT 75,
  eventtrim integer DEFAULT 7,
  alarmtrim integer DEFAULT 30,
  sensorreadingtrim integer DEFAULT 30,
  locationhistorytrim integer DEFAULT 90,
  commandtrim integer DEFAULT 365,
  sensorstattrim integer DEFAULT 30,
  smtpemailenablessl boolean DEFAULT false,
  smtpemailfrom text,
  smtpemailhost text,
  smtpemailpassword text,
  smtpemailport integer DEFAULT 25,
  smtpemailto text,
  smtpemailusername text,
  CONSTRAINT pk_applicationsetting_id PRIMARY KEY (id)
)
WITH (
  OIDS=FALSE
);
ALTER TABLE public.applicationsetting
  OWNER TO postgres;

CREATE TABLE public.changelog
(
  id serial NOT NULL,
  tablename text,
  recordid integer,
  changetype text,
  changedate timestamp with time zone NOT NULL DEFAULT now(),
  changedby text,
  CONSTRAINT pk_changelog_id PRIMARY KEY (id)
)
WITH (
  OIDS=FALSE
);
ALTER TABLE public.changelog
  OWNER TO postgres;

CREATE TABLE public.configurationchange
(
  id serial NOT NULL,
  modifiedtable text,
  changeevent text,
  recordid integer,
  created timestamp with time zone DEFAULT now(),
  madeby text NULL DEFAULT 'db_default'::text,
  CONSTRAINT pk_configurationchange_id PRIMARY KEY (id)
)
WITH (
  OIDS=FALSE
);
ALTER TABLE public.configurationchange
  OWNER TO postgres;

CREATE TABLE public.customfield
(
  name text NOT NULL,
  value text,
  display boolean NOT NULL DEFAULT false,
  displayindex integer NOT NULL DEFAULT '-1'::integer,
  uieditable boolean NOT NULL DEFAULT true,
  handhelddisplay boolean NOT NULL DEFAULT false,
  handheldvalue text,
  handhelddisplayindex integer NOT NULL DEFAULT '-1'::integer,
  datatype text,
  CONSTRAINT pk_customfield PRIMARY KEY (name)
)
WITH (
  OIDS=FALSE
);
ALTER TABLE public.customfield
  OWNER TO postgres;

CREATE TABLE public.customfieldvalue
(
  value text NOT NULL,
  displayindex integer,
  customfieldname text NOT NULL,
  CONSTRAINT fk_customfieldvalue_customfield FOREIGN KEY (customfieldname)
      REFERENCES public.customfield (name) MATCH SIMPLE
      ON UPDATE NO ACTION ON DELETE CASCADE
)
WITH (
  OIDS=FALSE
);
ALTER TABLE public.customfieldvalue
  OWNER TO postgres;

CREATE TABLE public.event
(
  id serial NOT NULL,
  eventtime timestamp with time zone NOT NULL DEFAULT now(),
  rfidtag text,
  readername text,
  readerlocation text,
  assetname text,
  assetdescription text,
  eventtype text,
  thresholdtype text,
  eventdetails text,
  filterannotation text,
  antennanumber text,
  rssi text,
  locationname text,
  vtagid text,
  CONSTRAINT pk_event_id PRIMARY KEY (id)
)
WITH (
  OIDS=FALSE
);
ALTER TABLE public.event
  OWNER TO postgres;

CREATE TABLE public.filedata
(
  data text,
  id serial NOT NULL ,
  CONSTRAINT pk_filedata PRIMARY KEY (id)
)
WITH (
  OIDS=FALSE
);
ALTER TABLE public.filedata
  OWNER TO postgres;

CREATE TABLE public.location
(
  id serial NOT NULL,
  name text,
  site text,
  building text,
  floor text,
  room text,
  description text,
  rfidtag text,
  lastmodifiedby text,
  CONSTRAINT pk_location_id PRIMARY KEY (id)
)
WITH (
  OIDS=FALSE
);
ALTER TABLE public.location
  OWNER TO postgres;

CREATE TABLE public.mapsite
(
  id serial NOT NULL,
  name text,
  originx numeric,
  originy numeric,
  CONSTRAINT pk_mapsite_id PRIMARY KEY (id)
)
WITH (
  OIDS=FALSE
);
ALTER TABLE public.mapsite
  OWNER TO postgres;

CREATE TABLE public.maparea
(
  id serial NOT NULL,
  floorlevel integer NOT NULL,
  name text NOT NULL,
  pixelspermeter numeric NOT NULL,
  mapsiteid integer NOT NULL,
  filedataid integer,
  imagewidth integer,
  imageheight integer,
  CONSTRAINT pk_maparea_id PRIMARY KEY (id),
  CONSTRAINT fk_maparea_filedata FOREIGN KEY (filedataid)
      REFERENCES public.filedata (id) MATCH SIMPLE
      ON UPDATE NO ACTION ON DELETE SET NULL,
  CONSTRAINT fk_maparea_mapsite FOREIGN KEY (mapsiteid)
      REFERENCES public.mapsite (id) MATCH SIMPLE
      ON UPDATE NO ACTION ON DELETE CASCADE
)
WITH (
  OIDS=FALSE
);
ALTER TABLE public.maparea
  OWNER TO postgres;


CREATE TABLE public.plugin
(
  id serial NOT NULL,
  name text,
  description text,
  filedataid integer NOT NULL,
  lastmodifiedby text,
  CONSTRAINT pk_plugin_id PRIMARY KEY (id),
  CONSTRAINT fk_plugin_filedata FOREIGN KEY (filedataid)
      REFERENCES public.filedata (id) MATCH SIMPLE
      ON UPDATE NO ACTION ON DELETE NO ACTION
)
WITH (
  OIDS=FALSE
);
ALTER TABLE public.plugin
  OWNER TO postgres;

CREATE TABLE public.sensorreading
(
  id serial NOT NULL,
  vtagid text,
  name text,
  created timestamp with time zone NOT NULL DEFAULT now(),
  sourcedistancehops text,
  ttl integer,
  minimumtemperaturedegc numeric,
  maximumtemperaturedegc numeric,
  minimumhumiditypercent numeric,
  maximumhumiditypercent numeric,
  batteryvolts numeric,
  x numeric,
  y numeric,
  z smallint,
  locationconfidence text,
  maximumaccelerationg numeric,
  maximumlightlevellux numeric,
  CONSTRAINT pk_sensorreading PRIMARY KEY (id)
)
WITH (
  OIDS=FALSE
);
ALTER TABLE public.sensorreading
  OWNER TO postgres;

CREATE TABLE public.sensorstat
(
  id serial NOT NULL,
  vtagid text,
  created timestamp with time zone NOT NULL DEFAULT now(),
  sourcedistancehops text,
  ttl integer,
  numneighbors integer,
  numactiveneighbors integer,
  totalerrors integer,
  lasterror integer,
  bestneighbor text,
  bestneighborrssidbm integer,
  minqueue integer,
  maxqueue integer,
  avgqueue integer,
  processedpackets integer,
  alarmcount integer,
  firmwareversion integer,
  movementtriggers integer,
  suspecttriggers text,
  dayssincereset numeric,
  leafdroppedpackets integer,
  CONSTRAINT pk_sensorstat PRIMARY KEY (id)
)
WITH (
  OIDS=FALSE
);
ALTER TABLE public.sensorstat
  OWNER TO postgres;

CREATE TABLE public.serverstatus
(
  name text NOT NULL,
  lastseen timestamp with time zone NOT NULL DEFAULT now(),
  CONSTRAINT pk_serverstatus PRIMARY KEY (name)
)
WITH (
  OIDS=FALSE
);
ALTER TABLE public.serverstatus
  OWNER TO postgres;

CREATE TABLE public.session
(
  id serial NOT NULL,
  ticketid text,
  name text,
  lastmodified timestamp with time zone NOT NULL DEFAULT now()
)
WITH (
  OIDS=FALSE
);
ALTER TABLE public.session
  OWNER TO postgres;

CREATE TABLE public.sysuser
(
  username text NOT NULL,
  password text,
  firstname text,
  lastname text,
  email text,
  phone text,
  cardid text,
  hideadminpopups boolean NOT NULL DEFAULT false,
  viewonly boolean NOT NULL DEFAULT false,
  rfidtag text,
  id serial NOT NULL,
  CONSTRAINT pk_sysuser PRIMARY KEY (id)
)
WITH (
  OIDS=FALSE
);
ALTER TABLE public.sysuser
  OWNER TO postgres;

CREATE TABLE public.template
(
  id serial NOT NULL,
  name text,
  templatetype text,
  filename text,
  usewithservice text,
  CONSTRAINT pk_template_id PRIMARY KEY (id)
)
WITH (
  OIDS=FALSE
);
ALTER TABLE public.template
  OWNER TO postgres;

CREATE TABLE public.vtag
(
  vtagid text,
  vtagx numeric,
  vtagy numeric,
  vtagz smallint,
  vtagpositiontype text,
  vtaglastseen timestamp with time zone,
  batterylevel numeric,
  readerid integer
)
WITH (
  OIDS=FALSE
);
ALTER TABLE public.vtag
  OWNER TO postgres;

CREATE TABLE public.printjob
(
  id serial NOT NULL,
  recordid integer NOT NULL,
  tablename text NOT NULL,
  templateid integer NOT NULL,
  generatedrfid text,
  completed boolean NOT NULL DEFAULT false,
  created timestamp with time zone NOT NULL DEFAULT now(),
  CONSTRAINT pk_printjob PRIMARY KEY (id),
  CONSTRAINT fk_printjob_template FOREIGN KEY (templateid)
      REFERENCES public.template (id) MATCH SIMPLE
      ON UPDATE NO ACTION ON DELETE CASCADE
)
WITH (
  OIDS=FALSE
);
ALTER TABLE public.printjob
  OWNER TO postgres;

CREATE TABLE public.reader
(
  id serial NOT NULL,
  name text,
  readermodel text,
  radsmodeenabled boolean DEFAULT false,
  port text,
  portstring text,
  devicesettings text,
  connectionstatus text,
  usewithservice text,
  locationid integer NOT NULL,
  assetmovedalertingactionid integer,
  lowbatteryalertingactionid integer,
  sensorthresholdalertingactionid integer,
  innerdirectionalertingactionid integer,
  outerdirectionalertingactionid integer,
  mapsiteid integer,
  readerdisconnectedalertingactionid integer,
  lastmodifiedby text,
  CONSTRAINT pk_reader_id PRIMARY KEY (id),
  CONSTRAINT fk_reader_alertingaction1 FOREIGN KEY (assetmovedalertingactionid)
      REFERENCES public.alertingaction (id) MATCH SIMPLE
      ON UPDATE NO ACTION ON DELETE NO ACTION,
  CONSTRAINT fk_reader_alertingaction2 FOREIGN KEY (lowbatteryalertingactionid)
      REFERENCES public.alertingaction (id) MATCH SIMPLE
      ON UPDATE NO ACTION ON DELETE NO ACTION,
  CONSTRAINT fk_reader_alertingaction3 FOREIGN KEY (sensorthresholdalertingactionid)
      REFERENCES public.alertingaction (id) MATCH SIMPLE
      ON UPDATE NO ACTION ON DELETE NO ACTION,
  CONSTRAINT fk_reader_alertingaction4 FOREIGN KEY (innerdirectionalertingactionid)
      REFERENCES public.alertingaction (id) MATCH SIMPLE
      ON UPDATE NO ACTION ON DELETE NO ACTION,
  CONSTRAINT fk_reader_alertingaction5 FOREIGN KEY (outerdirectionalertingactionid)
      REFERENCES public.alertingaction (id) MATCH SIMPLE
      ON UPDATE NO ACTION ON DELETE NO ACTION,
  CONSTRAINT fk_reader_alertingaction6 FOREIGN KEY (readerdisconnectedalertingactionid)
      REFERENCES public.alertingaction (id) MATCH SIMPLE
      ON UPDATE NO ACTION ON DELETE SET NULL,
  CONSTRAINT fk_reader_location FOREIGN KEY (locationid)
      REFERENCES public.location (id) MATCH SIMPLE
      ON UPDATE NO ACTION ON DELETE NO ACTION,
  CONSTRAINT fk_reader_mapsite FOREIGN KEY (mapsiteid)
      REFERENCES public.mapsite (id) MATCH SIMPLE
      ON UPDATE NO ACTION ON DELETE NO ACTION
)
WITH (
  OIDS=FALSE
);
ALTER TABLE public.reader
  OWNER TO postgres;

CREATE TABLE public.antenna
(
  id serial NOT NULL,
  "number" integer NOT NULL,
  alertingactionid integer NOT NULL,
  locationid integer NOT NULL,
  readerid integer NOT NULL,
  lastmodifiedby text,
  CONSTRAINT pk_antenna_id PRIMARY KEY (id),
  CONSTRAINT fk_antenna_alertingaction FOREIGN KEY (alertingactionid)
      REFERENCES public.alertingaction (id) MATCH SIMPLE
      ON UPDATE NO ACTION ON DELETE NO ACTION,
  CONSTRAINT fk_antenna_location FOREIGN KEY (locationid)
      REFERENCES public.location (id) MATCH SIMPLE
      ON UPDATE NO ACTION ON DELETE NO ACTION,
  CONSTRAINT fk_antenna_reader FOREIGN KEY (readerid)
      REFERENCES public.reader (id) MATCH SIMPLE
      ON UPDATE NO ACTION ON DELETE NO ACTION
)
WITH (
  OIDS=FALSE
);
ALTER TABLE public.antenna
  OWNER TO postgres;

CREATE TABLE public.asset
(
  id serial NOT NULL,
  name text,
  description text,
  rfidtag text,
  assettype text,
  departmentcode text,
  maxunseentime integer,
  lastobservedlocation text,
  lastobservedtime timestamp with time zone DEFAULT now(),
  checkinstatus text,
  checkedoutto text,
  additionalinformation text,
  disposalstatus text DEFAULT 'In Service'::text,
  disposalmethod text,
  disposaldate date,
  disposaldestination text,
  maintenancestartdate date,
  maintenancesingledate date,
  nextmaintenance date,
  date1 date,
  date2 date,
  date3 date,
  date4 date,
  date5 date,
  listvalue1 text,
  listvalue2 text,
  listvalue3 text,
  listvalue4 text,
  listvalue5 text,
  text1 text,
  text2 text,
  text3 text,
  text4 text,
  text5 text,
  text6 text,
  text7 text,
  text8 text,
  text9 text,
  text10 text,
  text11 text,
  text12 text,
  text13 text,
  text14 text,
  text15 text,
  text16 text,
  text17 text,
  text18 text,
  text19 text,
  text20 text,
  maintenancemethod text,
  maintenanceintervalmonths integer,
  nearestfixed text,
  lastmodified timestamp with time zone NOT NULL DEFAULT now(),
  locationid integer,
  sensorreadinghistoryid integer,
  alertingactionid integer,
  sensorstatshistoryid integer,
  readerid integer,
  assetparentid integer,
  assetchildcount integer,
  vtagboxx numeric,
  vtagboxy numeric,
  vtagboxwidth numeric,
  vtagboxheight numeric,
  vtagx numeric,
  vtagy numeric,
  vtagaccelsensor boolean,
  vtagpositiontype text,
  vtagalgorithmtype text,
  batterylevel numeric,
  vtagid text,
  vtagz smallint,
  vtaglastseen timestamp with time zone,
  vtaglastmoved timestamp with time zone,
  filedataid integer,
  created timestamp with time zone NOT NULL DEFAULT now(),
  lastmodifiedby text,
  CONSTRAINT pk_asset_id PRIMARY KEY (id),
  CONSTRAINT fk_asset_alertingaction FOREIGN KEY (alertingactionid)
      REFERENCES public.alertingaction (id) MATCH SIMPLE
      ON UPDATE NO ACTION ON DELETE SET NULL,
  CONSTRAINT fk_asset_asset FOREIGN KEY (assetparentid)
      REFERENCES public.asset (id) MATCH SIMPLE
      ON UPDATE NO ACTION ON DELETE SET NULL,
  CONSTRAINT fk_asset_location FOREIGN KEY (locationid)
      REFERENCES public.location (id) MATCH SIMPLE
      ON UPDATE NO ACTION ON DELETE NO ACTION,
  CONSTRAINT fk_asset_reader FOREIGN KEY (readerid)
      REFERENCES public.reader (id) MATCH SIMPLE
      ON UPDATE NO ACTION ON DELETE SET NULL
)
INHERITS (public.vtag)
WITH (
  OIDS=FALSE
);
ALTER TABLE public.asset
  OWNER TO postgres;

CREATE TABLE public.checkouthistory
(
  id serial NOT NULL,
  checkinstatus text,
  individual text,
  transactiontime timestamp with time zone NOT NULL DEFAULT now(),
  assetid integer NOT NULL,
  CONSTRAINT pk_checkouthistory_id PRIMARY KEY (id),
  CONSTRAINT fk_checkouthistory_asset FOREIGN KEY (assetid)
      REFERENCES public.asset (id) MATCH SIMPLE
      ON UPDATE NO ACTION ON DELETE CASCADE
)
WITH (
  OIDS=FALSE
);
ALTER TABLE public.checkouthistory
  OWNER TO postgres;

CREATE TABLE public.fixedvtag
(
  name text,
  vtagid text NOT NULL,
  vtagx numeric,
  vtagy numeric,
  vtagz smallint,
  vtagpositiontype text,
  vtaglastseen timestamp with time zone,
  batterylevel numeric,
  readerid integer,
  locationid integer,
  id serial NOT NULL,
  lastmodifiedby text,
  CONSTRAINT pk_fixedvtag PRIMARY KEY (id),
  CONSTRAINT fk_fixedvtag_location FOREIGN KEY (locationid)
      REFERENCES public.location (id) MATCH SIMPLE
      ON UPDATE NO ACTION ON DELETE SET NULL,
  CONSTRAINT fx_fixedvtag_reader FOREIGN KEY (readerid)
      REFERENCES public.reader (id) MATCH SIMPLE
      ON UPDATE NO ACTION ON DELETE SET NULL
)
INHERITS (public.vtag)
WITH (
  OIDS=FALSE
);
ALTER TABLE public.fixedvtag
  OWNER TO postgres;

CREATE TABLE public.locationhistory
(
  id serial NOT NULL,
  timeseen timestamp with time zone NOT NULL DEFAULT now(),
  assetid integer NOT NULL,
  locationid integer,
  CONSTRAINT pk_locationhistory_id PRIMARY KEY (id),
  CONSTRAINT fk_locationhistory_asset FOREIGN KEY (assetid)
      REFERENCES public.asset (id) MATCH SIMPLE
      ON UPDATE NO ACTION ON DELETE CASCADE,
  CONSTRAINT fk_locationhistory_location FOREIGN KEY (locationid)
      REFERENCES public.location (id) MATCH SIMPLE
      ON UPDATE NO ACTION ON DELETE CASCADE
)
WITH (
  OIDS=FALSE
);
ALTER TABLE public.locationhistory
  OWNER TO postgres;

CREATE TABLE public.maintenancehistory
(
  id serial NOT NULL,
  whenperformed timestamp with time zone NOT NULL DEFAULT now(),
  actionperformed text,
  notes text,
  performedby text,
  assetid integer NOT NULL,
  CONSTRAINT pk_maintenancehistory_id PRIMARY KEY (id),
  CONSTRAINT fk_maintenancehistory_asset FOREIGN KEY (assetid)
      REFERENCES public.asset (id) MATCH SIMPLE
      ON UPDATE NO ACTION ON DELETE CASCADE
)
WITH (
  OIDS=FALSE
);
ALTER TABLE public.maintenancehistory
  OWNER TO postgres;

CREATE TABLE public.queuedcommand
(
  id serial NOT NULL,
  vtagid text,
  commandtype text,
  commandtext text,
  status text,
  lastattempt timestamp with time zone,
  created timestamp with time zone NOT NULL DEFAULT now(),
  reason text,  
  retrycount integer NOT NULL DEFAULT 0,
  readerid integer,
  CONSTRAINT pk_queuedcommand_id PRIMARY KEY (id),  
  CONSTRAINT fk_queuedcommand_reader FOREIGN KEY (readerid)
      REFERENCES public.reader (id) MATCH SIMPLE
      ON UPDATE NO ACTION ON DELETE SET NULL
)
WITH (
  OIDS=FALSE
);
ALTER TABLE public.queuedcommand
  OWNER TO postgres;

CREATE OR REPLACE VIEW public.v_sensorreading AS 
select appSetting.TemperatureFormat, sr.id,sr.vtagid,sr.name,sr.created,sr.sourcedistancehops,sr.ttl,sr.minimumhumiditypercent,sr.maximumhumiditypercent,sr.batteryvolts,sr.z,sr.locationconfidence,sr.maximumaccelerationg,sr.maximumlightlevellux,
CASE WHEN UseMeters=true THEN x ELSE ROUND(x * 3.28, 2) END AS x,
CASE WHEN UseMeters=true THEN y ELSE ROUND(y * 3.28, 2) END AS y,
CASE WHEN TemperatureFormat='Fahrenheit' THEN ROUND(minimumtemperaturedegc * 9 / 5 + 32, 2) ELSE minimumtemperaturedegc END AS minimumtemperaturedegc,
CASE WHEN TemperatureFormat='Fahrenheit' THEN ROUND(maximumtemperaturedegc * 9 / 5 + 32, 2) ELSE maximumtemperaturedegc END AS maximumtemperaturedegc
 from sensorreading sr CROSS JOIN applicationsetting AS appSetting

CREATE OR REPLACE VIEW public.v_asset AS 
 SELECT a.*,
    COALESCE(l.name, 'Missing'::text) AS locationname
   FROM asset a
     LEFT JOIN location l ON l.id = a.locationid;

ALTER TABLE public.v_asset
  OWNER TO postgres;

CREATE OR REPLACE VIEW public.v_assetvtag AS 
 SELECT a.*,
    COALESCE(l.name, 'Missing'::text) AS locationname
   FROM asset a
     LEFT JOIN location l ON l.id = a.locationid
  WHERE NOT a.vtagid IS NULL
  ORDER BY a.vtagid;

ALTER TABLE public.v_assetvtag
  OWNER TO postgres;

CREATE OR REPLACE VIEW public.v_batterylevel AS 
 SELECT a.name AS assetname,
    vt.*
   FROM vtag vt
     LEFT JOIN asset a ON a.vtagid = vt.vtagid
  ORDER BY vt.batterylevel;

ALTER TABLE public.v_batterylevel
  OWNER TO postgres;

CREATE OR REPLACE VIEW public.v_checkouthistory AS 
 SELECT a.name AS assetname,
    ch.*
   FROM asset a,
    checkouthistory ch
  WHERE a.id = ch.assetid;

ALTER TABLE public.v_checkouthistory
  OWNER TO postgres;

CREATE OR REPLACE VIEW public.v_fixedvtag AS 
 SELECT COALESCE(l.name, 'Missing'::text) AS locationname,
    r.name AS readername,
    vt.*
   FROM fixedvtag vt
     LEFT JOIN location l ON l.id = vt.locationid
     LEFT JOIN reader r ON r.id = vt.readerid
  ORDER BY vt.vtagid;

ALTER TABLE public.v_fixedvtag
  OWNER TO postgres;

CREATE OR REPLACE VIEW public.v_locationhistory AS 
 SELECT lh.*,
    a.name AS assetname,
    a.description AS assetdescription,
    COALESCE(l.name, 'Missing'::text) AS locationname
   FROM locationhistory lh
     LEFT JOIN location l ON l.id = lh.locationid
     LEFT JOIN asset a ON a.id = lh.assetid
  ORDER BY lh.assetid, lh.timeseen DESC;

ALTER TABLE public.v_locationhistory
  OWNER TO postgres;

CREATE OR REPLACE VIEW public.v_maintenancehistory AS 
 SELECT mh.*,
    a.name AS assetname,
    a.description AS assetdescription
   FROM asset a,
    maintenancehistory mh
  WHERE a.id = mh.assetid;

ALTER TABLE public.v_maintenancehistory
  OWNER TO postgres;

CREATE OR REPLACE VIEW public.v_site AS 
 SELECT DISTINCT location.site
   FROM location
  WHERE NOT location.site = NULL::text;

ALTER TABLE public.v_site
  OWNER TO postgres;

CREATE OR REPLACE VIEW public.v_vtag AS 
 SELECT vtag.*
   FROM vtag
  WHERE NOT vtag.vtagid IS NULL
  ORDER BY vtag.vtagid;

ALTER TABLE public.v_vtag
  OWNER TO postgres;
 
CREATE OR REPLACE FUNCTION process_insert_delete_configuration_change() RETURNS TRIGGER AS $config_audit$ 
   BEGIN 
       IF(TG_OP = 'DELETE') THEN 
           INSERT INTO configurationchange(modifiedtable, changeevent, recordid, madeby) SELECT TG_ARGV[0], 'Delete', OLD.id, 'db_trigger';
           RETURN OLD; 
       ELSIF(TG_OP = 'UPDATE') THEN 
           INSERT INTO configurationchange(modifiedtable, changeevent, recordid, madeby) SELECT TG_ARGV[0], 'Edit', NEW.id, NEW.lastmodifiedby; 
           RETURN NEW;  
       ELSIF(TG_OP = 'INSERT') THEN 
           INSERT INTO configurationchange(modifiedtable, changeevent, recordid, madeby) SELECT TG_ARGV[0], 'Add', NEW.id, NEW.lastmodifiedby;
           RETURN NEW; 
       END IF; 
       RETURN NULL; --result is ignored since this is an AFTER trigger 
   END;
$config_audit$ LANGUAGE plpgsql;

CREATE TRIGGER reader_insert_delete_configuration_change
AFTER INSERT OR UPDATE OR DELETE ON reader
    FOR EACH ROW EXECUTE PROCEDURE process_insert_delete_configuration_change('Reader');

CREATE TRIGGER alertingaction_insert_delete_configuration_change
AFTER INSERT OR UPDATE OR DELETE ON alertingaction
    FOR EACH ROW EXECUTE PROCEDURE process_insert_delete_configuration_change('AlertingAction');

CREATE TRIGGER alarmack_insert_delete_configuration_change
AFTER INSERT OR UPDATE OR DELETE ON alarmack
    FOR EACH ROW EXECUTE PROCEDURE process_insert_delete_configuration_change('AlarmAck');

CREATE TRIGGER location_insert_delete_configuration_change
AFTER INSERT OR UPDATE OR DELETE ON location
    FOR EACH ROW EXECUTE PROCEDURE process_insert_delete_configuration_change('Location');

CREATE TRIGGER fixedvtag_insert_delete_configuration_change
AFTER INSERT OR UPDATE OR DELETE ON fixedvtag
    FOR EACH ROW EXECUTE PROCEDURE process_insert_delete_configuration_change('FixedVTag');

CREATE TRIGGER asset_insert_delete_configuration_change
AFTER INSERT OR UPDATE OR DELETE ON asset
    FOR EACH ROW EXECUTE PROCEDURE process_insert_delete_configuration_change('Asset');

CREATE TRIGGER applicationsetting_insert_delete_configuration_change
AFTER INSERT OR UPDATE OR DELETE ON applicationsetting
  FOR EACH ROW EXECUTE PROCEDURE public.process_insert_delete_configuration_change('ApplicationSetting');

CREATE TRIGGER plugin_insert_delete_configuration_change
AFTER INSERT OR UPDATE OR DELETE ON plugin
  FOR EACH ROW EXECUTE PROCEDURE public.process_insert_delete_configuration_change('Plugin');

CREATE OR REPLACE FUNCTION process_reader_delete() RETURNS TRIGGER AS $reader_delete$
    BEGIN        
        IF (TG_OP = 'DELETE') THEN
	    UPDATE asset SET lastmodifiedby='db_default',readerid=null WHERE readerid=OLD.id;            
	    UPDATE fixedvtag SET lastmodifiedby='db_default',readerid=null WHERE readerid=OLD.id;
            RETURN OLD;               
        END IF;
        RETURN NULL; -- result is ignored since this is an AFTER trigger
    END;
$reader_delete$ LANGUAGE plpgsql;

CREATE TRIGGER reader_delete
BEFORE DELETE ON reader
    FOR EACH ROW EXECUTE PROCEDURE process_reader_delete();

CREATE OR REPLACE FUNCTION process_alertingaction_delete() RETURNS TRIGGER AS $alertingaction_delete$
    BEGIN        
        IF (TG_OP = 'DELETE') THEN
	    UPDATE asset SET lastmodifiedby='db_default',alertingactionid=null WHERE alertingaction=OLD.id;            
	    UPDATE reader SET lastmodifiedby='db_default' WHERE assetmovedalertingactionid=OLD.id OR lowbatteryalertingactionid=OLD.id OR sensorthresholdalertingactionid=OLD.id OR innerdirectionalertingactionid=OLD.id
		OR outerdirectionalertingactionid=OLD.id OR readerdisconnectedalertingactionid=OLD.id;
            RETURN OLD;               
        END IF;
        RETURN NULL; -- result is ignored since this is an AFTER trigger
    END;
$alertingaction_delete$ LANGUAGE plpgsql;

CREATE TRIGGER alertingaction_delete
BEFORE DELETE ON alertingaction
    FOR EACH ROW EXECUTE PROCEDURE process_alertingaction_delete();

CREATE OR REPLACE FUNCTION process_location_delete() RETURNS TRIGGER AS $location_delete$
    BEGIN        
        IF (TG_OP = 'DELETE') THEN
			UPDATE asset SET lastmodifiedby='db_default',locationid=null WHERE locationid=OLD.id;            
			UPDATE fixedvtag SET lastmodifiedby='db_default',locationid=null WHERE locationid=OLD.id;            
			UPDATE reader SET lastmodifiedby='db_default',locationid=null WHERE locationid=OLD.id;            	    			
            RETURN OLD;             
        END IF;
        RETURN NULL; -- result is ignored since this is an AFTER trigger
    END;
$location_delete$ LANGUAGE plpgsql;

CREATE TRIGGER location_delete
BEFORE DELETE ON location
    FOR EACH ROW EXECUTE PROCEDURE process_location_delete();

CREATE OR REPLACE FUNCTION process_reader_delete() RETURNS TRIGGER AS $reader_delete$
    BEGIN        
        IF (TG_OP = 'DELETE') THEN
	    UPDATE asset SET lastmodifiedby='db_default',readerid=null WHERE readerid=OLD.id;            
	    UPDATE fixedvtag SET lastmodifiedby='db_default',readerid=null WHERE readerid=OLD.id;            	    
            RETURN OLD;               
        END IF;
        RETURN NULL; -- result is ignored since this is an AFTER trigger
    END;
$reader_delete$ LANGUAGE plpgsql;

CREATE TRIGGER reader_delete
BEFORE DELETE ON location
    FOR EACH ROW EXECUTE PROCEDURE process_reader_delete();
